/*
 This file is part of GNU Taler
 (C) 2021-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 */

import { AmountString, TalerMerchantApi } from "@gnu-taler/taler-util";
import { addDays } from "date-fns";
import { FunctionalComponent, h } from "preact";
import { DetailPage as TestedComponent } from "./DetailPage.js";

export default {
  title: "Pages/Order/Detail",
  component: TestedComponent,
  argTypes: {
    onRefund: { action: "onRefund" },
    onBack: { action: "onBack" },
  },
};

function createExample<Props>(
  Component: FunctionalComponent<Props>,
  props: Partial<Props>,
) {
  const r = (args: any) => <Component {...args} />;
  r.args = props;
  return r;
}

const defaultContractTerm: TalerMerchantApi.ContractTerms = {
  amount: "TESTKUDOS:10" as AmountString,
  timestamp: {
    t_s: new Date().getTime() / 1000,
  },
  exchanges: [],
  max_fee: "TESTKUDOS:1" as AmountString,
  merchant: {} as any,
  merchant_base_url: "http://merchant.url/",
  order_id: "2021.165-03GDFC26Y1NNG",
  products: [],
  summary: "text summary",
  wire_transfer_deadline: {
    t_s: "never",
  },
  refund_deadline: { t_s: "never" },
  merchant_pub: "ASDASDASDSd",
  nonce: "QWEQWEQWE",
  pay_deadline: {
    t_s: "never",
  },
  wire_method: "x-taler-bank",
  h_wire: "asd",
};

// contract_terms: defaultContracTerm,
export const Claimed = createExample(TestedComponent, {
  id: "2021.165-03GDFC26Y1NNG",
  selected: {
    order_status: "claimed",
    contract_terms: defaultContractTerm,
  },
});

export const PaidNotRefundable = createExample(TestedComponent, {
  id: "2021.165-03GDFC26Y1NNG",
  selected: {
    order_status: "paid",
    contract_terms: defaultContractTerm,
    refunded: false,
    deposit_total: "TESTKUDOS:10" as AmountString,
    exchange_code: 0,
    order_status_url: "http://merchant.backend/status",
    exchange_http_status: 0,
    refund_amount: "TESTKUDOS:0" as AmountString,
    refund_details: [],
    refund_pending: false,
    wire_details: [],
    wired: false,
    wire_reports: [],
  },
});

export const PaidRefundable = createExample(TestedComponent, {
  id: "2021.165-03GDFC26Y1NNG",
  selected: {
    order_status: "paid",
    contract_terms: {
      ...defaultContractTerm,
      refund_deadline: {
        t_s: addDays(new Date(), 2).getTime() / 1000,
      },
    },
    refunded: false,
    deposit_total: "TESTKUDOS:10" as AmountString,
    exchange_code: 0,
    order_status_url: "http://merchant.backend/status",
    exchange_http_status: 0,
    refund_amount: "TESTKUDOS:0" as AmountString,
    refund_details: [],
    wire_reports: [],
    refund_pending: false,
    wire_details: [],
    wired: false,
  },
});

export const Unpaid = createExample(TestedComponent, {
  id: "2021.165-03GDFC26Y1NNG",
  selected: {
    order_status: "unpaid",
    order_status_url: "http://merchant.backend/status",
    creation_time: {
      t_s: new Date().getTime() / 1000,
    },
    summary: "text summary",
    taler_pay_uri: "pay uri",
    total_amount: "TESTKUDOS:10" as AmountString,
  },
});
